"use client"

import { useState, useRef, useEffect } from "react"
import { SearchBar } from "@/components/search-bar"
import { ResultPanel } from "@/components/result-panel"
import { SearchHistory } from "@/components/search-history"
import { Header } from "@/components/header"
import { ThemeProvider } from "@/components/theme-provider"

interface SearchHistoryItem {
  id: string
  query: string
  result: string
  timestamp: number
  duration: number
}

export default function HomePage() {
  const [query, setQuery] = useState("")
  const [result, setResult] = useState("")
  const [isLoading, setIsLoading] = useState(false)
  const [hasSearched, setHasSearched] = useState(false)
  const [searchHistory, setSearchHistory] = useState<SearchHistoryItem[]>([])
  const [searchStartTime, setSearchStartTime] = useState<number>(0)
  const [searchDuration, setSearchDuration] = useState<number>(0)
  const resultRef = useRef<HTMLDivElement>(null)

  // Load search history from localStorage on mount
  useEffect(() => {
    const savedHistory = localStorage.getItem("searchgpt-history")
    if (savedHistory) {
      try {
        setSearchHistory(JSON.parse(savedHistory))
      } catch (error) {
        console.error("Error loading search history:", error)
      }
    }
  }, [])

  // Save search history to localStorage
  const saveToHistory = (searchQuery: string, searchResult: string, duration: number) => {
    const newItem: SearchHistoryItem = {
      id: Date.now().toString(),
      query: searchQuery,
      result: searchResult,
      timestamp: Date.now(),
      duration,
    }

    const updatedHistory = [newItem, ...searchHistory].slice(0, 10) // Keep only last 10 searches
    setSearchHistory(updatedHistory)
    localStorage.setItem("searchgpt-history", JSON.stringify(updatedHistory))
  }

  const handleSearch = async (searchQuery: string) => {
    if (!searchQuery.trim()) return

    setQuery(searchQuery)
    setResult("")
    setIsLoading(true)
    setHasSearched(true)
    const startTime = Date.now()
    setSearchStartTime(startTime)
    setSearchDuration(0)

    try {
      // System prompt for SearchGPT
      const systemPrompt =
        "You are SearchGPT — a powerful AI search assistant. Format your response using clean Markdown with structured headings, links, images, and citations. Act like a combination of ChatGPT, Google, and Perplexity. Summarize, extract facts, and always provide sources when possible."

      // Use puter.ai.chat with streaming
      const response = await (window as any).puter.ai.chat(`${systemPrompt}\n\nUser Query: ${searchQuery}`, {
        model: "openrouter:openai/gpt-4o-search-preview",
        stream: true,
      })

      let fullResponse = ""

      // Stream the response
      for await (const chunk of response) {
        if (chunk?.text) {
          fullResponse += chunk.text
          setResult(fullResponse)

          // Auto-scroll to latest content
          if (resultRef.current) {
            resultRef.current.scrollTop = resultRef.current.scrollHeight
          }
        }
      }

      // Calculate final duration and save to history
      const finalDuration = Math.round((Date.now() - startTime) / 1000)
      setSearchDuration(finalDuration)
      saveToHistory(searchQuery, fullResponse, finalDuration)
    } catch (error) {
      console.error("Search error:", error)
      const errorMessage = "Sorry, there was an error processing your search. Please try again."
      setResult(errorMessage)
      const finalDuration = Math.round((Date.now() - startTime) / 1000)
      setSearchDuration(finalDuration)
      saveToHistory(searchQuery, errorMessage, finalDuration)
    } finally {
      setIsLoading(false)
    }
  }

  // Load puter.js on component mount
  useEffect(() => {
    const script = document.createElement("script")
    script.src = "https://js.puter.com/v2/"
    script.async = true
    document.head.appendChild(script)

    return () => {
      document.head.removeChild(script)
    }
  }, [])

  // Update duration in real-time while searching
  useEffect(() => {
    let interval: NodeJS.Timeout
    if (isLoading && searchStartTime > 0) {
      interval = setInterval(() => {
        const currentDuration = Math.round((Date.now() - searchStartTime) / 1000)
        setSearchDuration(currentDuration)
      }, 100) // Update every 100ms for smoother counting
    }
    return () => {
      if (interval) clearInterval(interval)
    }
  }, [isLoading, searchStartTime])

  return (
    <ThemeProvider>
      <div className="min-h-screen bg-gray-900 text-white">
        <Header />

        <main className="container mx-auto px-4 py-8">
          <div className={`transition-all duration-500 ${hasSearched ? "mt-8" : "mt-32"}`}>
            <SearchBar onSearch={handleSearch} isLoading={isLoading} hasSearched={hasSearched} />
          </div>

          {hasSearched && (
            <div className="mt-8" ref={resultRef}>
              <ResultPanel
                query={query}
                result={result}
                isLoading={isLoading}
                duration={searchDuration}
                startTime={searchStartTime}
              />
            </div>
          )}

          {searchHistory.length > 0 && (
            <div className="mt-12">
              <SearchHistory history={searchHistory} onSearch={handleSearch} />
            </div>
          )}
        </main>
      </div>
    </ThemeProvider>
  )
}
