"use client"

import { useEffect, useState } from "react"
import { Card } from "@/components/ui/card"
import { ScrollArea } from "@/components/ui/scroll-area"
import { Button } from "@/components/ui/button"
import { ChevronDown, ChevronUp, Copy, Check, Clock } from "lucide-react"
import { marked } from "marked"
import Image from "next/image"

interface ResultPanelProps {
  query: string
  result: string
  isLoading: boolean
  duration: number
  startTime: number
}

export function ResultPanel({ query, result, isLoading, duration, startTime }: ResultPanelProps) {
  const [htmlContent, setHtmlContent] = useState("")
  const [copied, setCopied] = useState(false)
  const [isExpanded, setIsExpanded] = useState(false)

  useEffect(() => {
    if (result) {
      // Configure marked for better rendering
      marked.setOptions({
        breaks: true,
        gfm: true,
      })

      const html = marked(result)
      setHtmlContent(html)
    }
  }, [result])

  // Auto-expand when loading is complete
  useEffect(() => {
    if (!isLoading && result) {
      setIsExpanded(true)
    }
  }, [isLoading, result])

  const handleCopy = async () => {
    try {
      await navigator.clipboard.writeText(result)
      setCopied(true)
      setTimeout(() => setCopied(false), 2000)
    } catch (err) {
      console.error("Failed to copy text:", err)
    }
  }

  const toggleExpanded = () => {
    setIsExpanded(!isExpanded)
  }

  return (
    <div className="space-y-4">
      {/* Query Display */}
      <div className="text-lg font-medium text-gray-200 mb-4 flex items-center space-x-3">
        <span>
          Search Results for: <span className="text-blue-400">"{query}"</span>
        </span>
      </div>

      {/* Collapsed Response Box */}
      <Card className="bg-gray-800/50 border-gray-700 backdrop-blur-sm overflow-hidden transition-all duration-500 ease-in-out">
        {/* Header */}
        <div
          className="p-4 cursor-pointer hover:bg-gray-700/30 transition-colors duration-200 flex items-center justify-between"
          onClick={toggleExpanded}
        >
          <div className="flex items-center space-x-3">
            <div className="flex items-center space-x-2">
              <Clock className="h-4 w-4 text-blue-400" />
              <span className="text-sm text-gray-300">
                {isLoading ? (
                  <span className="flex items-center space-x-1">
                    <span>Searching</span>
                    <span className="animate-pulse">...</span>
                    <span className="text-blue-400">{duration}s</span>
                  </span>
                ) : (
                  <span>
                    Searched for <span className="text-blue-400">{duration}s</span>
                  </span>
                )}
              </span>
            </div>
          </div>

          <div className="flex items-center space-x-2">
            {result && !isLoading && (
              <Button
                variant="ghost"
                size="sm"
                onClick={(e) => {
                  e.stopPropagation()
                  handleCopy()
                }}
                className="text-gray-400 hover:text-white"
              >
                {copied ? <Check className="h-4 w-4 text-green-400" /> : <Copy className="h-4 w-4" />}
              </Button>
            )}
            <Button variant="ghost" size="sm" className="text-gray-400 hover:text-white">
              {isExpanded ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
            </Button>
          </div>
        </div>

        {/* Expandable Content */}
        <div
          className={`transition-all duration-500 ease-in-out ${
            isExpanded ? "max-h-[800px] opacity-100" : "max-h-0 opacity-0"
          } overflow-hidden`}
        >
          <div className="px-4 pb-4">
            <div className="text-sm text-gray-400 mb-3 flex items-center space-x-2">
              <span>{isExpanded && !isLoading ? "Collapse details" : "Expand for details"}</span>
            </div>

            {/* Fade Effect Container */}
            <div className="relative">
              {/* Top Fade */}
              <div className="absolute top-0 left-0 right-0 h-4 bg-gradient-to-b from-gray-800/50 to-transparent z-10 pointer-events-none" />

              {/* Content */}
              <ScrollArea className="h-[600px] w-full">
                <div className="pr-4 py-2">
                  {isLoading && !result && (
                    <div className="space-y-3">
                      <div className="animate-pulse h-4 bg-gray-700 rounded w-3/4"></div>
                      <div className="animate-pulse h-4 bg-gray-700 rounded w-1/2"></div>
                      <div className="animate-pulse h-4 bg-gray-700 rounded w-5/6"></div>
                    </div>
                  )}

                  {result && (
                    <div
                      className="prose prose-invert prose-blue max-w-none
                        prose-headings:text-gray-200 
                        prose-p:text-gray-300 prose-p:leading-relaxed
                        prose-a:text-blue-400 prose-a:no-underline hover:prose-a:underline
                        prose-strong:text-gray-200
                        prose-code:text-pink-400 prose-code:bg-gray-900 prose-code:px-1 prose-code:rounded
                        prose-pre:bg-gray-900 prose-pre:border prose-pre:border-gray-700
                        prose-blockquote:border-l-blue-400 prose-blockquote:text-gray-300
                        prose-ul:text-gray-300 prose-ol:text-gray-300
                        prose-li:text-gray-300"
                      dangerouslySetInnerHTML={{ __html: htmlContent }}
                    />
                  )}

                  {isLoading && result && (
                    <div className="mt-4 flex items-center space-x-2 text-blue-400">
                      <div className="animate-pulse h-2 w-2 bg-blue-400 rounded-full"></div>
                      <span className="text-sm">Continuing search...</span>
                    </div>
                  )}
                </div>
              </ScrollArea>

              {/* Bottom Fade */}
              <div className="absolute bottom-0 left-0 right-0 h-4 bg-gradient-to-t from-gray-800/50 to-transparent z-10 pointer-events-none" />
            </div>
          </div>
        </div>

        {/* Collapsed Preview */}
        {!isExpanded && result && !isLoading && (
          <div className="px-4 pb-4">
            <div className="text-sm text-gray-400 line-clamp-2">{result.substring(0, 150)}...</div>
          </div>
        )}
      </Card>
    </div>
  )
}
