"use client"

import type React from "react"
import { useState } from "react"
import { Search, Loader2 } from "lucide-react"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import Image from "next/image"

interface SearchBarProps {
  onSearch: (query: string) => void
  isLoading: boolean
  hasSearched: boolean
}

export function SearchBar({ onSearch, isLoading, hasSearched }: SearchBarProps) {
  const [query, setQuery] = useState("")

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    if (query.trim() && !isLoading) {
      onSearch(query.trim())
    }
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault()
      handleSubmit(e)
    }
  }

  return (
    <div className={`w-full max-w-4xl mx-auto transition-all duration-500 ${hasSearched ? "" : "transform scale-110"}`}>
      {!hasSearched && (
        <div className="text-center mb-8">
          <h2 className="text-4xl font-bold mb-4 bg-gradient-to-r from-blue-400 via-purple-400 to-pink-400 bg-clip-text text-transparent">
            What would you like to search for?
          </h2>
          <p className="text-gray-400 text-lg">
            Ask me anything and I'll search the web to give you comprehensive answers
          </p>

          {/* Powered by OpenAI branding */}
          <div className="mt-4 flex items-center justify-center space-x-2 text-sm text-gray-500">
            <span>Powered by</span>
            <Image src="/openai-text.png" alt="OpenAI" width={50} height={16} className="object-contain opacity-70" />
            <Image
              src="/openai-logo.png"
              alt="OpenAI Logo"
              width={16}
              height={16}
              className="object-contain opacity-70"
            />
          </div>
        </div>
      )}

      <form onSubmit={handleSubmit} className="relative">
        <div className="relative">
          <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 h-5 w-5" />
          <Input
            type="text"
            placeholder="Search for anything..."
            value={query}
            onChange={(e) => setQuery(e.target.value)}
            onKeyPress={handleKeyPress}
            className="w-full pl-12 pr-24 py-4 text-lg bg-gray-800 border-gray-700 focus:border-blue-500 focus:ring-blue-500 rounded-xl"
            disabled={isLoading}
          />
          <Button
            type="submit"
            disabled={!query.trim() || isLoading}
            className="absolute right-2 top-1/2 transform -translate-y-1/2 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-700 p-3 rounded-lg"
          >
            {isLoading ? <Loader2 className="h-5 w-5 animate-spin" /> : <Search className="h-5 w-5" />}
          </Button>
        </div>
      </form>

      {!hasSearched && (
        <div className="mt-8 flex flex-wrap gap-2 justify-center">
          {[
            "What's happening in AI today?",
            "Explain quantum computing",
            "Best programming languages 2024",
            "Climate change solutions",
            "Space exploration news",
          ].map((suggestion) => (
            <button
              key={suggestion}
              onClick={() => {
                setQuery(suggestion)
                onSearch(suggestion)
              }}
              className="px-4 py-2 bg-gray-800 hover:bg-gray-700 rounded-full text-sm transition-colors border border-gray-700 hover:border-gray-600"
              disabled={isLoading}
            >
              {suggestion}
            </button>
          ))}
        </div>
      )}
    </div>
  )
}
