"use client"

import { useState } from "react"
import { Card } from "@/components/ui/card"
import { ScrollArea } from "@/components/ui/scroll-area"
import { Button } from "@/components/ui/button"
import { ChevronDown, ChevronUp, Copy, Check, Clock, Search } from "lucide-react"
import { marked } from "marked"

interface SearchHistoryItem {
  id: string
  query: string
  result: string
  timestamp: number
  duration: number
}

interface SearchHistoryProps {
  history: SearchHistoryItem[]
  onSearch: (query: string) => void
}

export function SearchHistory({ history, onSearch }: SearchHistoryProps) {
  const [expandedItems, setExpandedItems] = useState<Set<string>>(new Set())
  const [copiedItems, setCopiedItems] = useState<Set<string>>(new Set())

  const toggleExpanded = (id: string) => {
    const newExpanded = new Set(expandedItems)
    if (newExpanded.has(id)) {
      newExpanded.delete(id)
    } else {
      newExpanded.add(id)
    }
    setExpandedItems(newExpanded)
  }

  const handleCopy = async (id: string, result: string) => {
    try {
      await navigator.clipboard.writeText(result)
      const newCopied = new Set(copiedItems)
      newCopied.add(id)
      setCopiedItems(newCopied)
      setTimeout(() => {
        setCopiedItems((prev) => {
          const updated = new Set(prev)
          updated.delete(id)
          return updated
        })
      }, 2000)
    } catch (err) {
      console.error("Failed to copy text:", err)
    }
  }

  const formatTimestamp = (timestamp: number) => {
    const date = new Date(timestamp)
    const now = new Date()
    const diffInHours = Math.floor((now.getTime() - date.getTime()) / (1000 * 60 * 60))

    if (diffInHours < 1) {
      const diffInMinutes = Math.floor((now.getTime() - date.getTime()) / (1000 * 60))
      return diffInMinutes < 1 ? "Just now" : `${diffInMinutes}m ago`
    } else if (diffInHours < 24) {
      return `${diffInHours}h ago`
    } else {
      const diffInDays = Math.floor(diffInHours / 24)
      return `${diffInDays}d ago`
    }
  }

  if (history.length === 0) {
    return null
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center space-x-3">
        <Search className="h-5 w-5 text-gray-400" />
        <h2 className="text-xl font-semibold text-gray-200">Recent Searches</h2>
        <div className="h-px bg-gray-700 flex-1" />
      </div>

      <div className="space-y-4">
        {history.map((item) => {
          const isExpanded = expandedItems.has(item.id)
          const isCopied = copiedItems.has(item.id)
          const htmlContent = marked(item.result)

          return (
            <Card
              key={item.id}
              className="bg-gray-800/30 border-gray-700 backdrop-blur-sm overflow-hidden transition-all duration-500 ease-in-out"
            >
              {/* Header */}
              <div
                className="p-4 cursor-pointer hover:bg-gray-700/30 transition-colors duration-200"
                onClick={() => toggleExpanded(item.id)}
              >
                <div className="flex items-center justify-between">
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center space-x-3 mb-2">
                      <Clock className="h-4 w-4 text-blue-400 flex-shrink-0" />
                      <span className="text-sm text-gray-300">
                        Searched for <span className="text-blue-400">{item.duration}s</span>
                      </span>
                      <span className="text-xs text-gray-500">{formatTimestamp(item.timestamp)}</span>
                    </div>
                    <h3 className="text-base font-medium text-gray-200 truncate">{item.query}</h3>
                    {!isExpanded && (
                      <p className="text-sm text-gray-400 mt-1 line-clamp-2">{item.result.substring(0, 120)}...</p>
                    )}
                  </div>

                  <div className="flex items-center space-x-2 ml-4">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={(e) => {
                        e.stopPropagation()
                        onSearch(item.query)
                      }}
                      className="text-gray-400 hover:text-blue-400 transition-colors"
                      title="Search again"
                    >
                      <Search className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={(e) => {
                        e.stopPropagation()
                        handleCopy(item.id, item.result)
                      }}
                      className="text-gray-400 hover:text-white transition-colors"
                      title="Copy result"
                    >
                      {isCopied ? <Check className="h-4 w-4 text-green-400" /> : <Copy className="h-4 w-4" />}
                    </Button>
                    <Button variant="ghost" size="sm" className="text-gray-400 hover:text-white transition-colors">
                      {isExpanded ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
                    </Button>
                  </div>
                </div>
              </div>

              {/* Expandable Content */}
              <div
                className={`transition-all duration-500 ease-in-out ${
                  isExpanded ? "max-h-[600px] opacity-100" : "max-h-0 opacity-0"
                } overflow-hidden`}
              >
                <div className="px-4 pb-4">
                  <div className="text-sm text-gray-400 mb-3">
                    {isExpanded ? "Collapse details" : "Expand for details"}
                  </div>

                  {/* Fade Effect Container */}
                  <div className="relative">
                    {/* Top Fade */}
                    <div className="absolute top-0 left-0 right-0 h-4 bg-gradient-to-b from-gray-800/30 to-transparent z-10 pointer-events-none" />

                    {/* Content */}
                    <ScrollArea className="h-[400px] w-full">
                      <div className="pr-4 py-2">
                        <div
                          className="prose prose-invert prose-blue max-w-none prose-sm
                            prose-headings:text-gray-200 
                            prose-p:text-gray-300 prose-p:leading-relaxed
                            prose-a:text-blue-400 prose-a:no-underline hover:prose-a:underline
                            prose-strong:text-gray-200
                            prose-code:text-pink-400 prose-code:bg-gray-900 prose-code:px-1 prose-code:rounded
                            prose-pre:bg-gray-900 prose-pre:border prose-pre:border-gray-700
                            prose-blockquote:border-l-blue-400 prose-blockquote:text-gray-300
                            prose-ul:text-gray-300 prose-ol:text-gray-300
                            prose-li:text-gray-300"
                          dangerouslySetInnerHTML={{ __html: htmlContent }}
                        />
                      </div>
                    </ScrollArea>

                    {/* Bottom Fade */}
                    <div className="absolute bottom-0 left-0 right-0 h-4 bg-gradient-to-t from-gray-800/30 to-transparent z-10 pointer-events-none" />
                  </div>
                </div>
              </div>
            </Card>
          )
        })}
      </div>
    </div>
  )
}
